/* ==================================================================
File        : mainFilter.hlsl
Author      : various
Date        : 2007
Format      : HLSL/Cg
Description : Shaders used by the 'main' filter, bloom, DoF, etc

================================================================== */

#include "default.hlsl"


//------------------------------------------------------------------------------------------------
// Defines
//------------------------------------------------------------------------------------------------

//#define PRESERVE_HDR_SATURATION		// Ensures bright colours have their saturation preserved rather than clamping to white
#define LUMINANCE_FROM_DOT_PRODUCT		// Calculates luminance from dot product such that white is brighter than red, green or blue
//#define LUMINANCE_FROM_MAX_CHANNEL	// Calculates luminance from maximum component such that white is just as bright as, red green and blue


//------------------------------------------------------------------------------------------------
// Texture samplers
//------------------------------------------------------------------------------------------------

//@:-texformat fullColor_tex RGBA8
//@:-texformat noiseTex RGBA8

sampler2D fullColor_tex				: register(s0);	/// full res color tex
sampler2D noiseTex					: register(s6);


//------------------------------------------------------------------------------------------------
// Shader constants
//------------------------------------------------------------------------------------------------


//------------------------------------------------------------------------------------------------
// Vertex output structs
//------------------------------------------------------------------------------------------------

struct VS_OUTMAINFILTER
{
	float4 position : POSITION;
	float4 texcoord : TEXCOORD0;
#if defined(SM1)
	float2 texcoord2 : TEXCOORD1;
#endif

#if defined(BLOOM)
	float4 bloomFactors : TEXCOORD2;
#endif

#if defined(DOF)
	float3 dofParams : TEXCOORD3;
#endif

#if defined(PROCESSCOLOUR) && !defined(SAT_ONLY)
	float4 noiseParams : TEXCOORD4;
	float4 offsets : TEXCOORD5;
#else
	#if defined(BLOOM)
		float4 offsets : TEXCOORD5;
	#endif
#endif

#if defined(COC)
	float4 CoCParams : TEXCOORD6;
#endif

};


//------------------------------------------------------------------------------------------------
// Post Processes
//------------------------------------------------------------------------------------------------

#include "mainFilterBloom.hlsl"
#include "mainFilterDepthOfField.hlsl"
#include "mainFilterProcessColour.hlsl"
#include "mainFilterWaterDistortion.hlsl"

//------------------------------------------------------------------------------------------------
// Calculate texture coordinates
//------------------------------------------------------------------------------------------------

half2 CalculateTexcoord( VS_IN_2D input )
{	
	half2 texcoords;
	
	#ifdef _WINPC
	texcoords.xy = input.uvSet0;
	#else
	texcoords.xy = input.position.xy * half2(0.5, -0.5) + 0.5;
	#endif
	
#if defined(WATERDISTORTION)
	return CalculateWaterDistortion(texcoords, input);
#else	
	return texcoords;
#endif
}

//------------------------------------------------------------------------------------------------
// Final Pass Vertex Shader
//------------------------------------------------------------------------------------------------

VS_OUTMAINFILTER mainfilter_vx(VS_IN_2D input)
{
	VS_OUTMAINFILTER output;
	
	output.position = input.position;
	
	output.texcoord.xy = CalculateTexcoord( input );

	output.texcoord.zw = output.texcoord.xy; 
	output.texcoord.xy = output.texcoord.xy * g_ViewportScaleBias.zw + g_ViewportScaleBias.xy;	

	#if defined(SM1)
		output.texcoord2.xy = output.texcoord.zw;
	#endif

#if defined(PROCESSCOLOUR) && !defined(SAT_ONLY)
	output.offsets = output.texcoord.zwzw * noiseParams.xyxy + noiseOffset;
	output.noiseParams.zw = noiseParams.zw;
	output.noiseParams.xy = noiseParams2.xy;
#else
	#if defined(BLOOM)
		output.offsets = output.texcoord.zwzw * noiseParams.xyxy + noiseOffset;
	#endif
#endif

#if defined(BLOOM)
	output.bloomFactors = bloomFactors;

	#if defined(BLOOM_SPOT)
		output.bloomFactors.r *= computeBloomRadianSpot(input);
	#else
		output.bloomFactors.a = computeBloomRadian(input);
		output.bloomFactors.r *= output.bloomFactors.a;
	#endif
#endif

#if defined(COC)
	output.CoCParams = CoCParams;
#endif

#if defined(DOF)
	output.dofParams = dofParams;
#endif

	return output;
}


//------------------------------------------------------------------------------------------------
// Final Pass Fragment Shader
//------------------------------------------------------------------------------------------------

float4 mainfilter_px( VS_OUTMAINFILTER In ) : COLOR
{
	float HDRRange = 1.0;

	// Read in source colour (still in compressed range)

	#if defined(PROCESSCOLOUR) && !defined(SAT_ONLY)
		half3 noise = ProcessColourNoise(In.offsets.xy, In.offsets.zw, In.noiseParams.z);
	#endif

	#if defined(PROCESSCOLOUR) && !defined(SAT_ONLY) && defined(PAINT)
//		half4 SourceColour = (
//			tex2D( fullColor_tex, In.texcoord.xy + noise.xy * In.noiseParams.x ) + 
//			tex2D( fullColor_tex, In.texcoord.xy + noise.yz * In.noiseParams.x ) +
//			tex2D( fullColor_tex, In.texcoord.xy + noise.zx * In.noiseParams.x ) + 
//			tex2D( fullColor_tex, In.texcoord.xy + noise.yx * In.noiseParams.x ) + 
//			tex2D( fullColor_tex, In.texcoord.xy + noise.zy * In.noiseParams.x ) +
//			tex2D( fullColor_tex, In.texcoord.xy + noise.xz * In.noiseParams.x )
//			) / 6.0;
		half4 SourceColour = tex2D( fullColor_tex, In.texcoord.xy + noise.xy * In.noiseParams.x );
	#else
		half4 SourceColour = tex2D( fullColor_tex, In.texcoord.xy );
	#endif

	// Determine texture coordinates to use
	half2 PostEffectTexCoords;
	#if defined(SM1)
		PostEffectTexCoords = In.texcoord2.xy;
	#else
		PostEffectTexCoords = In.texcoord.zw; // xy
	#endif

	#if defined(DOF)
		// Do new DOF blend
		float4 BlurColour = tex2D(mipColor_tex, PostEffectTexCoords);
		SourceColour.rgb = lerp(SourceColour.rgb, BlurColour.rgb, BlurColour.a);
		HDRRange = In.dofParams.z; // Get HDR range from DOF params (as bloom and colour processing may be disabled)
	#endif

#if defined(BLOOM)
	// Do bloom effect
	SourceColour = Bloom( SourceColour, PostEffectTexCoords, In.bloomFactors, In.offsets.xy );

	HDRRange = In.bloomFactors.z; // Get HDR range from Bloom params (as dof and colour processing may be disabled)
#endif

#if defined(PROCESSCOLOUR) && !defined(SAT_ONLY)
	HDRRange = In.noiseParams.y; // Get HDR range from colour prcessing params (as bloom and dof may be disabled)
#endif

	// Transform from HDR compressed range to LDR for display
	SourceColour.rgb = SourceColour.rgb * HDRRange;

#if defined(PROCESSCOLOUR)
	// Do colour processing on final LDR image
	#if defined(SAT_ONLY)
		SourceColour = ProcessColour( SourceColour, saturation.r );
	#else
		#if defined(PAINT)
			SourceColour = ProcessColourPaint( SourceColour, In.offsets.xy, In.noiseParams.w );
		#else
			SourceColour = ProcessColour( SourceColour, noise, In.noiseParams.w );
		#endif
	#endif
#endif

#if defined(PRESERVE_HDR_SATURATION)
	// Retain saturated colour
	half maxComponent = max( max( dstColor.r, dstColor.g ), dstColor.b );
	if (maxComponent > 1.0)
	{
		dstColor.rgb /= maxComponent;
	}
#endif

	return SourceColour;
}


//------------------------------------------------------------------------------------------------
// Exposure Control
//------------------------------------------------------------------------------------------------

float4 thresholdpass_px( VS_OUT_POS_UVSET01 In ) : COLOR
{
	float4 output;

	float4 col = tex2D(fullColor_tex, In.uvSet01.xy);

#if defined(LUMINANCE_FROM_DOT_PRODUCT)
	// Calculate intensity from dot product of colours with RGB weights
	// This is an attempt at physical accuracy (measuring true luminance)
	float Intensity = dot( col.rgb, thresholdPass.rgb );
#endif

#if defined(LUMINANCE_FROM_MAX_CHANNEL)
	// Calculate intensity as maximum component of colour after being scaled by RGB weights
	// This is not physically accurate but helps avoid a big difference in intensity between
	// solid colours (especially blue) and white
	col.rgb *= thresholdPass.rgb;
	float Intensity = max( max( col.r, col.g ), col.b );
#endif

#if defined(CG_PS3)
	// Clip using alpha-test
	output = float4( 0, 0, 0, Intensity - thresholdPass.w );
	return saturate( output );
#else
	// Clip using texkill
	output = float4(1,0,0,1);
	clip( Intensity - thresholdPass.w );
	return output;
#endif
}


//------------------------------------------------------------------------------------------------
// Edge Anti-aliasing Fragment Shader
//------------------------------------------------------------------------------------------------

struct VS_EDGE_IN_AA
{
	float4 position : POSITION;
	float2 uvSet0 : TEXCOORD0;
};

struct VS_EDGE_OUT_AA
{
	float4 position : POSITION;
	float4 uvSet01 : TEXCOORD0;
	float4 uvSet23 : TEXCOORD1;
	float2 uvSet4  : TEXCOORD2;
};

// x = -half_pixel_size.x
// y =  half_pixel_size.y
// z =  half_pixel_size.x
// w =  half_pixel_size.y
float4 half_pixel_offsets	: register( c20 );
float4 filter_params		: register( c0 );

VS_EDGE_OUT_AA edgeFilter_vx( VS_EDGE_IN_AA input )
{
	VS_EDGE_OUT_AA output;
	output.position = input.position;
	output.uvSet01.xy = input.uvSet0.xy;
	output.uvSet01.zw = input.uvSet0.xy + half2( -half_pixel_offsets.x,  half_pixel_offsets.y );
	output.uvSet23.xy = input.uvSet0.xy + half2(  half_pixel_offsets.x,  half_pixel_offsets.y );
	output.uvSet23.zw = input.uvSet0.xy - half2(  half_pixel_offsets.x,  half_pixel_offsets.y );
	output.uvSet4.xy  = input.uvSet0.xy - half2( -half_pixel_offsets.x,  half_pixel_offsets.y );
	return output;
}

float4 edgeFilter_px( VS_EDGE_OUT_AA input ) : COLOR
{
	// Sample the center.
	half4 c = tex2D(layer0_sampler, input.uvSet01.xy);

	// Sample a 3 by 3 grid.
	// Compute a summed grid value (useful in several palces below).
	half4 grid_summed = 0;
	grid_summed += tex2D(layer0_sampler, input.uvSet01.zw);
	grid_summed += tex2D(layer0_sampler, input.uvSet23.xy);
	grid_summed += tex2D(layer0_sampler, input.uvSet23.zw);
	grid_summed += tex2D(layer0_sampler, input.uvSet4.xy);

	// We have sampled the following...
	//
	// 1/4  1/2  1/4
	// 1/2   1   1/2
	// 1/4  1/2  1/4
	// 
	// And require a divide by 3 after subtracting 'c' to normalize.
	//
	half4 edge_intensity = (grid_summed - c) * 0.333333h;

	// Calculate edge difference. Scale gets it into 0 to >1 range.
	// Clamp to limit to 1, and also to stop neagtive turning positive when we square it.
	half4 temp1 = abs(c - edge_intensity);
	half4 edge_diff = saturate(filter_params.x * temp1);

	// Square to bias toward values near to 1.
	edge_diff *= edge_diff;

	// Average the 3 components.Switch to single channel calculations.
	half edge_diff_av = ( edge_diff.x + edge_diff.y + edge_diff.z ) * 0.333333h;
	
	// Clamp. Discard small differences. Assume it's noise.
	edge_diff_av *= step( filter_params.z, edge_diff_av );
	
	// Compute final colour as a lerp between the blurred and central pixel.
	half4 result = lerp( c, grid_summed * 0.25h, edge_diff_av * filter_params.w );
	
	// And some debugging stuff...
	return result;
//	return lerp( result, half4(edge_diff_av,edge_diff_av,edge_diff_av,1.0h), filter_params.y );
}
